---
title: process · Cloudflare Workers docs
description: The process module in Node.js provides a number of useful APIs
  related to the current process.
lastUpdated: 2025-09-19T20:05:01.000Z
chatbotDeprioritize: false
source_url:
  html: https://developers.cloudflare.com/workers/runtime-apis/nodejs/process/
  md: https://developers.cloudflare.com/workers/runtime-apis/nodejs/process/index.md
---

Note

To enable built-in Node.js APIs and polyfills, add the nodejs\_compat compatibility flag to your [Wrangler configuration file](https://developers.cloudflare.com/workers/wrangler/configuration/). This also enables nodejs\_compat\_v2 as long as your compatibility date is 2024-09-23 or later. [Learn more about the Node.js compatibility flag and v2](https://developers.cloudflare.com/workers/configuration/compatibility-flags/#nodejs-compatibility-flag).

The [`process`](https://nodejs.org/docs/latest/api/process.html) module in Node.js provides a number of useful APIs related to the current process.

Initially Workers only supported `nextTick`, `env`, `exit`, `getBuiltinModule`, `platform` and `features` on process, which was then updated with the [`enable_nodejs_process_v2`](https://developers.cloudflare.com/workers/configuration/compatibility-flags/#enable-process-v2-implementation) flag to include most Node.js process features.

Refer to the [Node.js documentation for `process`](https://nodejs.org/docs/latest/api/process.html) for more information.

Workers-specific implementation details apply when adapting Node.js process support for a serverless environment, which are described in more detail below.

## `process.env`

In the Node.js implementation of `process.env`, the `env` object is a copy of the environment variables at the time the process was started. In the Workers implementation, there is no process-level environment, so by default `env` is an empty object. You can still set and get values from `env`, and those will be globally persistent for all Workers running in the same isolate and context (for example, the same Workers entry point).

When [Node.js compatibility](https://developers.cloudflare.com/workers/runtime-apis/nodejs/) is turned on and the [`nodejs_compat_populate_process_env`](https://developers.cloudflare.com/workers/configuration/compatibility-flags/#enable-auto-populating-processenv) compatibility flag is set, `process.env` will contain any [environment variables](https://developers.cloudflare.com/workers/configuration/environment-variables/), [secrets](https://developers.cloudflare.com/workers/configuration/secrets/), or [version metadata](https://developers.cloudflare.com/workers/runtime-apis/bindings/version-metadata/) metadata that has been configured on your Worker.

### Relationship to per-request `env` argument in `fetch()` handlers

Workers have a concept of [environment variables](https://developers.cloudflare.com/workers/configuration/environment-variables/) that are applied on a per-Worker and per-request basis.

By default, these are not accessible via the `process.env` API. To automatically populate environment variables and secrets on `process.env`, enable the [`nodejs_compat_populate_process_env`](https://developers.cloudflare.com/workers/configuration/compatibility-flags/#nodejs_compat_populate_process_env) compatibility flag and disable the `disallow_importable_env` compatibility flag. It is also possible to manually copy these values into `process.env` if necessary -- but only within the context of a request.

Setting any value on `process.env` will coerce that value into a string.

```js
import * as process from "node:process";


export default {
  fetch(req, env) {
    // Set process.env.FOO to the value of env.FOO if process.env.FOO is not already set
    // and env.FOO is a string.
    process.env.FOO ??= (() => {
      if (typeof env.FOO === "string") {
        return env.FOO;
      }
    })();
  },
};
```

It is strongly recommended that you *do not* replace the entire `process.env` object with the cloudflare `env` object. Doing so will cause you to lose any environment variables that were set previously and will cause unexpected behavior for other Workers running in the same isolate. Specifically, it would cause inconsistency with the `process.env` object when accessed via named imports.

```js
import * as process from "node:process";
import { env } from "node:process";


process.env === env; // true! they are the same object
process.env = {}; // replace the object! Do not do this!
process.env === env; // false! they are no longer the same object


// From this point forward, any changes to process.env will not be reflected in env,
// and vice versa!
```

## `process.nextTick()`

The Workers implementation of `process.nextTick()` is a wrapper for the standard Web Platform API [`queueMicrotask()`](https://developer.mozilla.org/en-US/docs/Web/API/WindowOrWorkerGlobalScope/queueMicrotask).

```js
import { env, nextTick } from "node:process";


env["FOO"] = "bar";
console.log(env["FOO"]); // Prints: bar


nextTick(() => {
  console.log("next tick");
});
```

## Stdio

[`process.stdout`](https://nodejs.org/docs/latest/api/process.html#processstdout), [`process.stderr`](https://nodejs.org/docs/latest/api/process.html#processstderr) and [`process.stdin`](https://nodejs.org/docs/latest/api/process.html#processstdin) are supported as streams. `stdin` is treated as an empty readable stream. `stdout` and `stderr` are non-TTY writable streams, which output to normal logging output only with `stdout: `and `stderr: `prefixing.

The line buffer works by storing writes to stdout or stderr until either a newline character `\n` is encountered or until the next microtask, when the log is then flushed to the output.

This ensures compatibility with inspector and structured logging outputs.

## Current Working Directory

[`process.cwd()`](https://nodejs.org/docs/latest/api/process.html#processcwd) is the *current workding directory*, used as the default path for all filesystem operations, and is initialized to `/bundle`.

[`process.chdir()`](https://nodejs.org/docs/latest/api/process.html#processchdirdirectory) allows modifying the `cwd` and is respected by FS operations when using `enable_nodejs_fs_module`.

## Hrtime

While [`process.hrtime`](https://nodejs.org/docs/latest/api/process.html#processhrtimetime) high-resolution timer is available, it provides an inaccurate timer for compatibility only.
