---
title: Publish applications with Terraform · Cloudflare Learning Paths
description: This guide covers how to use the Cloudflare Terraform provider to
  quickly publish and secure a private application. In the following example, we
  will add a new published application to an existing Cloudflare Tunnel,
  configure how cloudflared proxies traffic to the application, and secure the
  application with Cloudflare Access.
lastUpdated: 2025-10-21T14:33:19.000Z
chatbotDeprioritize: false
source_url:
  html: https://developers.cloudflare.com/learning-paths/clientless-access/terraform/publish-apps-with-terraform/
  md: https://developers.cloudflare.com/learning-paths/clientless-access/terraform/publish-apps-with-terraform/index.md
---

This guide covers how to use the [Cloudflare Terraform provider](https://registry.terraform.io/providers/cloudflare/cloudflare/latest/docs) to quickly publish and secure a private application. In the following example, we will add a new published application to an existing Cloudflare Tunnel, configure how `cloudflared` proxies traffic to the application, and secure the application with Cloudflare Access.

## Prerequisites

* [Add your domain to Cloudflare](https://developers.cloudflare.com/learning-paths/clientless-access/initial-setup/add-site/)
* [Configure an IdP integration](https://developers.cloudflare.com/learning-paths/clientless-access/initial-setup/configure-idp/)
* [Create a Cloudflare Tunnel](https://developers.cloudflare.com/learning-paths/clientless-access/connect-private-applications/create-tunnel/#create-a-tunnel) via the Zero Trust dashboard
* Install the [Terraform client](https://developer.hashicorp.com/terraform/tutorials/aws-get-started/install-cli)
* [Create an API token](https://developers.cloudflare.com/fundamentals/api/get-started/create-token/) (refer to the [minimum required permissions](https://developers.cloudflare.com/cloudflare-one/networks/connectors/cloudflare-tunnel/deployment-guides/terraform/#3-create-a-cloudflare-api-token))

## 1. Create a Terraform configuration directory

Terraform functions through a working directory that contains configuration files. You can store your configuration in multiple files or just one — Terraform will evaluate all of the configuration files in the directory as if they were in a single document.

1. Create a folder for your Terraform configuration:

   ```sh
   mkdir cloudflare-tf
   ```

2. Change into the directory:

   ```sh
   cd cloudflare-tf
   ```

## 2. Declare providers and variables

Create a `.tf` file and copy-paste the following example. Fill in your API token, account and zone information, and Tunnel ID.

Find the Tunnel ID

1. In [Zero Trust](https://one.dash.cloudflare.com/), go to **Networks** > **Tunnels**.
2. Select the tunnel name.
3. Copy the **Tunnel ID**.

```txt
terraform {
  required_providers {
    cloudflare = {
      source = "cloudflare/cloudflare"
      version = "~> 4.0"
    }
  }
}


provider "cloudflare" {
  api_token = "<API-TOKEN>"
}


variable "account_id" {
  default = "<ACCOUNT-ID>"
}


variable "zone_id" {
  default = "<ZONE-ID>"
}


variable "zone_name" {
  default = "mycompany.com"
}


variable "tunnel_id" {
  default = "<TUNNEL-ID>"
}
```

Warning

To prevent accidentally exposing your Cloudflare credentials, do not save this file in your version control system. Learn more about [tracking a Terraform configuration](https://developers.cloudflare.com/terraform/tutorial/track-history/).

## 3. Configure Cloudflare resources

Add the following resources to your Terraform configuration.

### Add published application to Cloudflare Tunnel

Using the [`cloudflare_tunnel_config`](https://registry.terraform.io/providers/cloudflare/cloudflare/latest/docs/resources/tunnel_config) resource, create an ingress rule that maps your application to a public DNS record. This example makes `localhost:8080` available on `app.mycompany.com`, sets the [Connect Timeout](https://developers.cloudflare.com/cloudflare-one/networks/connectors/cloudflare-tunnel/configure-tunnels/cloudflared-parameters/origin-parameters/#connecttimeout), and enables [Access JWT validation](https://developers.cloudflare.com/cloudflare-one/networks/connectors/cloudflare-tunnel/configure-tunnels/cloudflared-parameters/origin-parameters/#access).

```txt
resource "cloudflare_tunnel_config" "example_config" {
  account_id = var.cloudflare_account_id
  tunnel_id  = var.tunnel_id


  config {
    ingress_rule {
      hostname = "app.${var.zone_name}"
      service  = "http://localhost:8080"
      origin_request {
        connect_timeout = "2m0s"
        access {
          required  = true
          team_name = "myteam"
          aud_tag   = [cloudflare_access_application.example_app.aud]
        }
      }
    }
    ingress_rule {
      # Respond with a `404` status code when the request does not match any of the previous hostnames.
      service  = "http_status:404"
    }
  }
}
```

Note

Published application configurations must include a catch-all ingress rule at the bottom of the file.

### Create an Access application

Using the [`cloudflare_access_application`](https://registry.terraform.io/providers/cloudflare/cloudflare/latest/docs/resources/access_application) resource, add the application to Cloudflare Access.

```txt
resource "cloudflare_access_application" "example_app" {
  zone_id                   = var.zone_id
  name                      = "Example application"
  domain                    = "app.${var.zone_name}"
  type                      = "self_hosted"
  session_duration          = "24h"
  auto_redirect_to_identity = false
}
```

### Create an Access policy

Using the [`cloudflare_access_policy`](https://registry.terraform.io/providers/cloudflare/cloudflare/latest/docs/resources/access_application) resource, create a policy to secure the application. The following policy will only allow access to users who authenticate through your identity provider.

```txt
resource "cloudflare_access_policy" "example_policy" {
  application_id    = cloudflare_access_application.example_app.id
  zone_id           = var.zone_id
  name              = "Example policy"
  precedence        = "1"
  decision          = "allow"


  include {
    login_method = ["<IDP-UUID>"]
  }


}
```

## 4. Deploy Terraform

To deploy the configuration files:

1. Initialize your configuration directory:

   ```sh
   terraform init
   ```

2. Preview everything that will be created:

   ```sh
   terraform plan
   ```

3. Apply the configuration:

   ```sh
   terraform apply
   ```

Users can now access the private application by going to the public URL and authenticating with Cloudflare Access. You can view your new tunnel route, Access application, and Access policy in [Zero Trust](https://one.dash.cloudflare.com). The new DNS record is shown in the [Cloudflare dashboard](https://dash.cloudflare.com).

Note

If you need to modify the Access application, Access policy or DNS record, you must make the changes via Terraform. Changes made via the dashboard will break Terraform's state. To prevent this from happening, [set the dashboard to read-only](https://developers.cloudflare.com/cloudflare-one/api-terraform/#set-dashboard-to-read-only).
